/* Collapsible side menu
 * The side menu can have sections that can be expanded or collapsed.
 * The path in the side menu to the current page is always expanded. After this, the state of the sections is retrieved
 * from session storage to keep the layout when navigating between pages. Any state that can't be retrieved from session
 * storage (like when vising the app for the first time) will use the application defaults.
 */
var sideMenu = {
    nodes: null,

    init: function() {
        //Build the mapping between the side menu dom elements and side menu tree data structure. This applies the
        //application defaults.
        this.nodes = new this.MenuNode($('#sub-nav').parent());

        //Apply any section states from the user's current visit.
        this.applyMenuClassesFromSessionStorage();

        //Expand the sections navigating to the current page, then apply the states to the page.
        $('#sub-nav li.active').parentsUntil('#sub-nav', '.collapsible').removeClass('collapsed').addClass('expanded');
        $('#sub-nav li.collapsed > ul').hide();
		$('#sub-nav li.collapsed > span > span.visuallyhidden').text('collapsed');
        $('#sub-nav li.expanded > ul').show();
		$('#sub-nav li.expanded > span > span.visuallyhidden').text('expanded');

        //When clicking a section heading without a link, swap the state of the section and store the new state in the
        //session storage.
        $('#sub-nav li.collapsible > span')
			.click(function(event) {
				//Don't run the handler when the menu is shrunk in responsive mode. See responsive.js.
				//i.e. if the responsivemenu object is defined and the menu is shrunk, skip the code.
				if(!(typeof responsivemenu != 'undefined' && responsivemenu.isMobileMenu && !responsivemenu.isMenuOpen)) {
					event.stopPropagation();
					sideMenu.toggleSection($(this).parent());
					sideMenu.saveMenuNodesToSessionStorage();
				}
			})
			.keydown(function(event) {
				//Don't run the handler when the menu is shrunk in responsive mode. See responsive.js.
				//i.e. if the responsivemenu object is defined and the menu is shrunk, skip the code.
				if(!(typeof responsivemenu != 'undefined' && responsivemenu.isMobileMenu && !responsivemenu.isMenuOpen)) {
					event.stopPropagation();
					console.log(event.which);
					if(event.which == 13) {
						//Key was enter
						sideMenu.toggleSection($(this).parent());
					} else if(event.which == 37) {
						//Key was left arrow
						sideMenu.collapseSection($(this).parent());
					} else if(event.which == 39) {
						//Key was right arrow
						sideMenu.expandSection($(this).parent());
					}
					sideMenu.saveMenuNodesToSessionStorage();
				}
			});
    },
	
	expandSection: function(liElement) {
		var section = $(liElement);
		if(section.is('.collapsed')) {
			section.removeClass('collapsed').addClass('expanded');
			$('> span > span.visuallyhidden', section).text('expanded');
			section.data('menuNode').collapsibleClass = 'expanded';
			section.children('ul').slideDown(200);
		}
	},
	
	collapseSection: function(liElement) {
		var section = $(liElement);
		if(section.is('.expanded')) {
			section.removeClass('expanded').addClass('collapsed');
			$('> span > span.visuallyhidden', section).text('collapsed');
			section.data('menuNode').collapsibleClass = 'collapsed';
			section.children('ul').slideUp(200);
		}
	},
	
	toggleSection: function(liElement) {
		var section = $(liElement);
		if(section.is('.collapsed')) {
			sideMenu.expandSection(liElement);
		} else if(section.is('.expanded')) {
			sideMenu.collapseSection(liElement);
		}
	},

    //Recursive constructor for the menu objects.
    MenuNode: function(liElement) {
        this.liElement = liElement;
        this.text = liElement.children('span, a').text();

        this.collapsibleClass = null;
        if(liElement.is('collapsed')) {
            this.collapsibleClass = 'collapsed';
        } else if(liElement.is('expanded')) {
            this.collapsibleClass = 'expanded';
        }

        this.children = {};
        var liChildren = $('> ul > li', liElement);
        for(var i = 0; i < liChildren.length; i++) {
            var liChild = liChildren.eq(i);
            var child = new sideMenu.MenuNode(liChild);
            this.children[child.text] = child;
            liChild.data('menuNode', child);
        }
    },

    applyMenuClassesFromSessionStorage: function() {
        //Get the tree from session storage. It should be the same as the nodes built from the dom structure but without
        //the references to the dom elements.
        if(!sessionStorage) {
            return;
        }
        var sessionTree = sessionStorage.getItem('sessionMenuNodes');
        if(!sessionTree) {
            return;
        }

        sessionTree = JSON.parse(sessionTree);
        this.recursivelyApplyMenuClassesFromSessionStorage(sessionTree, this.nodes);
    },

    //If collapsed or expanded, apply the state from the partial MenuNode from session storage to the full MenuNode from
    //and the DOM.
    recursivelyApplyMenuClassesFromSessionStorage: function (sessionTreeNode, nodesTreeNode) {
        if(sessionTreeNode.collapsibleClass == 'collapsed'
            || sessionTreeNode.collapsibleClass == 'expanded') {

            nodesTreeNode.collapsibleClass = sessionTreeNode.collapsibleClass;
            nodesTreeNode.liElement.removeClass('expanded collapsed').addClass(sessionTreeNode.collapsibleClass);
        }
        for(var childText in sessionTreeNode.children) {
            if(childText in nodesTreeNode.children) {
                this.recursivelyApplyMenuClassesFromSessionStorage(sessionTreeNode.children[childText], nodesTreeNode.children[childText]);
            }
        }
    },

    //Store the tree of MenuNodes to session storage.
    saveMenuNodesToSessionStorage: function() {
        if(!sessionStorage || !sideMenu.nodes) {
            return;
        }
        var sessionTree = sideMenu.nodes.toJSON();
        sessionStorage.setItem('sessionMenuNodes', sessionTree);
    }
};

//Serialise a MenuNode to JSON. We can't use JSON.stringify() on the children object because it causes an extra layer
//of string escaping every step down the tree.
sideMenu.MenuNode.prototype.toJSON = function() {
    var json = '{"text":' + JSON.stringify(this.text) +
        ',"collapsibleClass":' + JSON.stringify(this.collapsibleClass) +
        ',"children":{';
    var childStrings = [];
    for(var childText in this.children) {
        childStrings.push('"' + childText + '":' + this.children[childText].toJSON());
    }
    json = json + childStrings.join(',') + '}}';
    return json;
};

sideMenu.init();