/**
 * Very basic implementation of the observer pattern.
 * Subscribers can add and remove callbacks to specific
 * events, which will be fired when the publishers trigger
 * those events.
 */
(function($, global) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});

cti.userLogin.events = {

	/**
	 * Adds callbacks to events. These callbacks will be fired when
	 * publisher objects trigger the corresponding events.
	 *
	 * @param name The event's name (string).
	 * @param fn The callback function to be fired.
	 * @param bind The object `this` shall reference within the callback.
	 *
	 * @returns `this`
	 */
	on: function(name, fn, bind) {
		this._events || (this._events = {});
		this._events[name] || (this._events[name] = []);

		if ($.isFunction(fn)) {
			this._events[name].push({
				fn: fn,
				bind: bind
			});
		}

		return this;
	},

	/**
	 * Removes callbacks from an event. A specific callback can be
	 * unsubscribed, all callbacks for an event can be unsubscribed,
	 * or all callbacks on a specific event for a specific object can
	 * be removed.
	 *
	 * @param name The event name (string).
	 * @param fn The callback function to be removed. If not provided,
	 *           then all callbacks matching the `name` and `bind`
	 *           parameters will be removed.
	 * @param bind The value of the callback's `this`. If, for example
	 *             two different objects implement the exact same method,
	 *             and you only want to unsubscribe one of them, specifying
	 *             the `bind` parameter allows the other object's callback
	 *             to remain subscribed.
	 *
	 * @returns `this`
	 */
	off: function(name, fn, bind) {
		var callbacks = this._events && this._events[name],
			i = callbacks && callbacks.length - 1,
			obj;

		if (callbacks) {

			if (!fn && !bind) {
				callbacks.length = [];
			} else {
				while (i >= 0) {
					obj = callbacks[i];

					if ((!fn || obj.fn === fn) && (!bind || obj.bind === bind)) {
						callbacks.splice(0, 1);
					}

					i -= 1;
				}
			}
		}

		return this;
	},

	/**
	 * Fires all callbacks subscribed to a specific event.
	 *
	 * @param name The event's name
	 *
	 * @returns `this`
	 */
	trigger: function(name) {
		var callbacks = this._events && this._events[name],
			args;

		if (callbacks) {
			args = [].slice.call(arguments, 1);
			
			$.each(callbacks, function(i, callback) {
				
				if ($.isFunction(callback.fn)) {
					callback.fn.apply((callback.bind || null), args);
				}
			});
		}

		return this;
	}
};

})(window.Zepto || window.jQuery, window);
(function($, global, document) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});
cti.userLogin.login || (cti.userLogin.login = {});

var ns = cti.userLogin.login;

/**
 * Accepts an object of data, converts that data to HTML,
 * and returns the fragment.
 * The data must be in the following format:
 *
		{
			"element": "a",
			"text": "Lorem ipsum dolor sit amet.",
			"style": "font-weight: bold",
			"attributes": {
				"href": "/path/to/destination.html",
				"className": "clickableLink"
			},
			"children": []
		}
 *
 * Note that the data in the `attributes` object must be valid
 * element atttributes (e.g., `htmlFor`, `className`, etc.).
 *
 * This style is a hold-over from our MooTools days. It's not elegant,
 * and the JSON should be rewritten someday. But as that will take some
 * time, this will have to suffice for now.
 *
 * @param data The object of element data.
 *
 * @returns An HTML fragment ready to be inserted via `parent.appendChild`.
 */
ns.renderConfirmationHTML = (function() {
	function setElement(parent, obj) {
		var el = document.createElement(obj.element);

		if (obj.text) {
			el.appendChild(document.createTextNode(obj.text));
		}

		if (obj.style) {
			el.style.cssText = obj.style;
		}

		if (obj.attributes) {
			$.each(obj.attributes, function(name, value) {
				el[name] = value;
			});
		}

		if (obj.children) {
			obj.children.forEach(function(childObj) {
				setElement(el, childObj);
			});
		}

		parent.appendChild(el);
	}

	return function(data) {
		var frag = document.createDocumentFragment();

		data.forEach(function(obj) {
			setElement(frag, obj);
		});

		return frag;
	};
})();

/**
 * Generates the redirect URL from a given URL that will be used when users
 * log in. "logout=true" will be stripped from the query string.
 *
 * USAGE:
 * var url = 'http://www.example.com?a=b&logout=true&b=c#hash';
 * ns.getLoggedInUrl(url);
 * // returns: http://www.example.com?a=b&b=c#hash
 * 
 * @param url The starting URL (will almost always be `location.href`).
 * @return The new URL.
 */
ns.getLoggedInUrl = function(url) {
	return url.replace('logout=true', '').
			replace(/[&]+/, '&').
			replace(/\?[&]*($|#)/, '$1');
};

/**
 * Refreshes the current page, but strips out `logout=true`
 * from the query string.
 *
 * USAGE:
 * var url = 'http://www.example.com?a=b&logout=true&b=c#hash';
 * ns.refreshPage();
 * // redirects to: http://www.example.com?a=b&b=c#hash
 * var url = 'http://www.example.com?a=b#hash';
 * ns.refreshPage({ hash: 'newHash' });
 * // sets the new hash and refreshes the page.
 * 
 * @param options An object with overrides (currently only accepts `hash`).
 */
ns.refreshPage = function(options) {
	if (options && options.hash) {
		location.hash = options.hash;
	}
	
	if (location.href.indexOf('logout=true') !== -1) {
		location.href = ns.getLoggedInUrl(location.href);
	} else {
		location.reload();
	}
};

})(window.Zepto || window.jQuery, window, document);
/**
 * If this ever gets more complicated, then at some point it might make
 * sense to refactor this into separate MVC components. But right now
 * that seems a bit overkill.
 */
(function($, global, document) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});
cti.userLogin.login || (cti.userLogin.login = {});

var ns = cti.userLogin.login;

ns.accordion = {
	initialize: function(events, $container, $login, $register, $activate) {
		if (!$container) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing modal container in `accordion` controller.'
			};
		}

		if (!events) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing events object in `accordion` controller.'
			};
		}

		this.events = events;
		this.$container = $container;
		this.$folds = {
			login: $login,
			register: $register,
			activate: $activate
		};

		this.attach();
		this.subscribe();
	},

	attach: function() {
		this.$container.on('click', '.js-widget', $.proxy(function(e) {
			var doExpand = false,
				name = e.target.nodeName.toLowerCase(),
				$clicked;

			if (name === 'a' || name === 'img') {
				$clicked = $(e.target);

				if ($clicked.hasClass('js-toggler') ||
						$clicked.parent('.js-toggler').length) {

					doExpand = true;
				}
			} else if (name !== 'input') {
				doExpand = true;
			}

			if (doExpand) {
				e.preventDefault();
				this.expand($(e.currentTarget).data('name'));
			}
		}, this));
	},

	subscribe: function() {
		var accordion = this;
		
		function expandRegistrationForm() {
			accordion.expand('register');
		}

		if (this.$folds.register) {
			this.events.on('login.newUser', expandRegistrationForm, this);
			this.events.on('janrain.newUser', expandRegistrationForm, this);
		}
	},

	/**
	 * If there are both collapsed and expanded views:
	 * 1. Fade out collapsed
	 * 2. Slide to height of expanded
	 * 3. Add 'is-expanded' class to fold
	 * 4. Fade in expanded.
	 */
	expand: function(name) {
		var $fold = this.$folds[name],
			$expanded = $fold && $fold.find('.expandedView'),
			$collapsed = $fold && $fold.find('.collapsedView');

		if ($fold && !$fold.hasClass('is-expanded')) {

			if ($collapsed.length) {
				$collapsed.slideUp();
				$expanded.slideDown({
					complete: function() {
						$expanded.css('height', 'auto');
					}
				});
				$fold.removeClass('is-folded').addClass('is-expanded');
			} else {
				$expanded.slideDown({
					complete: function() {
						$fold.removeClass('is-folded').addClass('is-expanded');
						$expanded.css('height', 'auto');
					}
				});
			}

			this.collapseAll(name);
		}
	},

	/**
	 * 1. Fade out `.expandedView`
	 * 2. Slide up to height of `.collapsedView`
	 * 3. Add 'is-folded' class to fold
	 * 4. Fade in `.collapsedView`
	 */
	collapse: function(name) {
		var $fold = this.$folds[name],
			$expanded = $fold.find('.expandedView'),
			$collapsed = $fold.find('.collapsedView'),
			toHeight = $collapsed.length ? u$.height($collapsed) : 0;

		if ($fold && !$fold.hasClass('is-folded')) {

			// Login, activate: expanded & collapsed views
			// Register: expanded
			if ($collapsed.length) {
				$expanded.slideUp();
				$collapsed.slideDown({
					complete: function() {
						$collapsed.css('height', 'auto');
					}
				});
				$fold.removeClass('is-expanded').addClass('is-folded');
			} else {
				$expanded.slideUp({
					complete: function() {
						$fold.removeClass('is-expanded').addClass('is-folded');
					}
				});
			}
		}
	},

	collapseAll: function(name) {
		var key;

		for (key in this.$folds) {
			if (this.$folds.hasOwnProperty(key) && key !== name && this.$folds[key] !== null) {
				this.collapse(key);
			}
		}
	}
};

})(window.Zepto || window.jQuery, window, document);
(function($, global, document) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});
cti.userLogin.login || (cti.userLogin.login = {});

var ns = cti.userLogin.login;

ns.janrain = {
	initialize: function(events) {
		if (!events) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing events object in `janrain` controller'
			};
		}

		this.events = events;

		this.subscribe();
		this.setProviderFlow();
	},

	subscribe: function() {
		this.events.on('login.socialLogin', this.triggerFlow, this);
		this.events.on('register.socialRegister', this.triggerFlow, this);
	},

	setProviderFlow: function() {
		var ns = janrain.events.onProviderLoginToken;

		if (ns.eventHandlers && ns.eventHandlers.length) {
			ns.removeHandler(0);
		}

		ns.addHandler(function(response) {
			$.ajax({
				method: 'post',
				url: '/system/social/ajax/login.html',
				data: 'token=' + response.token,
				success: function(data, status, xhr) {
					var pattern = /<!--[\s\S]*-->/g,
						obj = $.parseJSON(xhr.responseText.replace(pattern, ''));

					this.routeResponse(obj);
				}.bind(this)
			});
		}.bind(this));
	},

	routeResponse: function(obj) {
		if (!obj.guest) {
			this.events.trigger('janrain.socialLogin', obj.providerName);
		} else {
			this.events.trigger('janrain.newUser', obj);
		}
	},

	triggerFlow: function(provider) {
		if (provider) {
			janrain.engage.signin.triggerFlow(provider);
		}
	}
};

})(window.Zepto || window.jQuery, window, document);
/**
 * The controller for the "login" section of the system login. Handles
 * logging in via a CT Network account, and also lost password retrieval.
 * If the user chooses to log in via a social media account, then control
 * is handed over to the Janrain controller (janrain.js);
 *
 * If this ever gets more complicated, then at some point it might make
 * sense to refactor this into separate MVC components. But right now
 * that seems a bit overkill.
 *
 * EVENTS FIRED:
 * login.needsToRegister (passed `email`)
 * login.socialLogin (passed `provider`)
 *
 * EVENTS WATCHED:
 * janrain.socialLogin (passed `provider`)
 */

// IN THE HTML, `.loginForm` REFERS TO THE `<div /` THAT CONTAINS
// THE LOGIN FORM.

(function($, global, document) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});
cti.userLogin.login || (cti.userLogin.login = {});

var ns = cti.userLogin.login,
	messages = {
		email: 'Please enter a valid email address',
		fail: 'Sorry, you are not logged in. Your email address and/or password' +
				' was invalid.',
		nonExistentEmail: 'There is no account associated with this email address.',
		success: 'You are now logged in and will be redirected in a moment.',
		retrievePassword: 'Your password has been sent to {{email}}.',
		ajaxError: 'Sorry, there was a problem with your request.' +
				' Please try again.'
	};

ns.login = $.extend(Object.create(u$.loaderMixin), {

	// This object exists solely to provide `u$.loaderMixin` with the
	// CSS class it should apply to the loader HTML.
	options: {
		loaderClass: 'loader-login'
	},

	/**
	 * Fetches the HTML elements associated with the login and password
	 * retrieval forms, then attaches submit and click events.
	 *
	 * @param $container The `$` HTML element container for the entire section.
	 * @param events The shared `cti.userLogin.events` object.
	 *
	 * @throws JanrainLoginError if no `$container` or `cti.userLogin.events`
	 *         object is provided.
	 */
	initialize: function($container, events) {
		if (!$container) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing widget container in `login` controller.'
			};
		}

		if (!events) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing events object in `login` controller.'
			};
		}

		this.events = events;

		this.setElements($container);
		this.subscribe();
		this.attach();
	},

	/**
	 * Sets the `$` HTML elements used by the application.
	 *
	 * @param $container The `$` HTML element container for the entire section.
	 */
	setElements: function($container) {
		this.$container = $container;
		this.$expanded = $container.find('.expandedView');
		
		this.$loginDiv = $container.find('.loginForm');
		this.$loginForm = this.$loginDiv.find('form');
		
		this.$passwordDiv = $container.find('.passwordRetrieval');
		this.$passwordForm = this.$passwordDiv.find('form');
		
	},

	/**
	 * `login` listens for the 'janrain.socialLogin' event,
	 * which is fired when users log in via a social media account.
	 */
	subscribe: function() {
		this.events.on('janrain.socialLogin', this.showLoginConfirmation, this);
	},

	/**
	 * Manages the following events:
	 * 1. Submit: Login Form
	 * 2. Submit: Forgot Password
	 * 3. Click: Forgot Password Link
	 * 4. Click: Social Media Links
	 */
	attach: function() {
		var onClick = $.proxy(function(e) {
				var $clicked = $(e.currentTarget);

				e.preventDefault();

				if ($clicked.hasClass('js-socialLogin')) {
					// attempt to log the user in via a social media account
					this.socialLogIn($clicked);
				} else {
					// toggle the login/password recovery form
					this.toggleForm();
				}
			}, this),
			onSubmit = $.proxy(function(e) {
				$('input.email').val(function(_, value) {
				   return $.trim(value);
				});
				$('input.password').val(function(_, value) {
		 			return $.trim(value);
				});
				var $form = $(e.currentTarget),
					action = $form.hasClass('js-recoverPassword') ? 'forgotpassword' :
							'login';

				e.preventDefault();

				// Send the password recovery email, or attempt to log the user in,
				// depending on the form element submitted.
				this.handleSubmit($form, action);
			}, this),
			delegates = '.js-socialLogin,.js-showRecoverPassword,.js-showLoginForm';

		this.$container.on('click', delegates, onClick).on('submit',
				'.js-login,.js-recoverPassword', onSubmit);
	},

	/**
	 * Hands control over the Janrain controller (not directly, but via
	 * the events object), passing it the provider name.
	 *
	 * @param $clicked The `$` HTML element that was clicked.
	 *
	 * @fires login.socialLogin (passed `provider`).
	 */
	socialLogIn: function($clicked) {
		var provider = $clicked.data('provider');

		this.events.trigger('login.socialLogin', provider);
	},

	/**
	 * Handles validation for the login and password retrieval events.
	 *
	 * @param data An array of data to validate. Should be in the same
	 *             format produced by the jQuery/Zepto `serializeArray` method.
	 *
	 * @returns Boolean `true` if the name is valid. Otherwise `false`.
	 */
	validate: (function() {
		var validators = {
			email: /^[A-Za-z0-9\-_\.]+@[A-Za-z0-9\-_\.]+$//*,
			password: /[^&\<\>]/g*/
		};

		return function(data) {
			var isValid = true;

			data.forEach(function(field) {
				var pattern = validators[field.name];

				if (pattern && !pattern.test(field.value)) {
					isValid = false;
				}
			});

			return isValid;
		};
	})(),

	setPasswordFormElements: function() {
		var $div;

		if (!this.$passwordElems) {
			$div = this.$passwordDiv;

			this.$passwordElems = {
				deck: $div.find('.deck'),
				form: $div.find('form'),
				logInNow: $div.find('.logInNow')
			};
		}

		return this.$passwordElems;
	},

	onShowPasswordForm: function() {
		var $elems = this.setPasswordFormElements(),
			deck = $elems.deck.data('text');

		if (deck) {
			$elems.deck.text(deck);
		}
		
		$elems.form.removeClass('is-invisible');	
		$elems.logInNow.addClass('is-invisible');
	},

	/**
	 * Displays either the login or password retrieval form, depending on
	 * which is currently hidden. Since only one will be displayed at any
	 * given time, there probably is no need for separate `showLoginForm`
	 * and `showPasswordRetrievalForm` methods.
	 */
	toggleForm: (function() {
		var forms = ['login', 'password'];

		return function() {
			// This may fall into the category of "too clever", but the above
			// `forms` array is used to keep track of which form should be
			// displayed.
			var $toShow = this['$' + forms[1] + 'Div'],
				$toHide = this['$' + forms[0] + 'Div'],
				showClass = 'is-' + forms[1] + 'FormShown',
				hideClass = 'is-' + forms[0] + 'FormShown';

			forms.unshift(forms.pop());

			// 1. Fade out the visible container.
			// 2. Hard set the container height as the visible container's height.
			// 3. Add `is-invisible` to the visible container.
			// 4. Set the hidden container's opacity to 0.
			// 5. Remove `is-invisible` from the hidden container.
			// 6. Fade in the hidden container.
			this.$container.removeClass(hideClass).addClass(showClass);
			this.$expanded.css('height', u$.height(this.$loginDiv));

			this.onShowPasswordForm();

			u$.slideUp($toHide, 0, {
				duration: 150,
				complete: function() {
					$toHide.addClass('is-invisible');

					u$.slideDown($toShow.removeClass('is-invisible'), null, {
						duration: 150,
						complete: function() {
							$toShow.css('height', 'auto');

							if ($toShow.hasClass('loginForm')) {
								this.$expanded.css('height', 'auto');
							}
						}.bind(this)
					});
				}.bind(this)
			});
		};
	})(),

	/**
	 * Validates the user data, and then hands control to the method
	 * responsible for posting user data. If validation fails, then
	 * an error message will be displayed to the user.
	 *
	 * @param $form The `$` form element submitted by the user.
	 * @param action The action associated with the form. Either 'login'
	 *               or 'forgotpassword'.
	 */
	handleSubmit: function($form, action) {
		if (!this.validate($form.serializeArray())) {
			this.showError(messages.email, $form);
		} else {
			this.hideError($form);
			this.send(action, $form.serialize());
		}
	},

	/**
	 * Displays an HTML error message to the user.
	 *
	 * @param message The string message to display.
	 * @param $form The `$` form element associated with the error.
	 */
	showError: function(message, $form) {
		var $error = $form.prev('.error');

		if (!$error.length) {
			$error = $('<p class="error" />');
			$form.before($error);
		}

		$error.css('opacity', 0).html(message).removeClass('is-invisible').
				animate({
					opacity: 1
				});
	},

	/**
	 * Hides an error message.
	 *
	 * @param $form The `$` form element associated with the error.
	 */
	hideError: function($form) {
		var $error = $form.prev('.error');

		if ($error.length) {
			$error.animate({
				opacity: 0
			}, {
				complete: function() {
					$error.addClass('is-invisible');
				}
			});
		}
	},

	/**
	 * Displays the default "AJAX Error" message, but also appends the
	 * error message provided by the XHR object assuming it's provided.
	 *
	 * @param action The action associated with request. Either
	 *               'passwordRetrieval' or 'login'.
	 * @param error The error message returned by the XHR object.
	 */
	showAjaxError: function(action, error) {
		var $form = action === 'passwordRetrieval' ? this.$passwordForm :
				this.$loginForm,
			message = messages.ajaxError;

		if (error) {
			message += '<br />Error: ' + error;
		}

		this.showError(message, $form);
	},

	/**
	 * Replaces the current `.expandedView` element with a confirmation
	 * message, and then refreshes the page.
	 *
	 * @param message The message to display to the user.
	 * @param refresh Should the page be refreshed?
	 */
	showConfirmation: function(message, refresh) {
		this.$container.removeClass('is-loginFormShown');
		u$.render(this.$container.find('.expandedView').html(''),
				'<p>' + message + '</p>');

		if (refresh) {
			// delay the redirect so that there is enough time to
			// register any Google Analytics events.
			setTimeout(function() {
				ns.refreshPage(ns._startData.redirectUrl);
			}.bind(this), 100);
		}
	},

	/**
	 * Displays a login confirmation message to the user, sends a
	 * login event to Google Analytics, and refreshes the page.
	 *
	 * @param provider The social media provider name to send to Google
	 *                 Analytics. If not used, "CT Network" is used instead.
	 */
	showLoginConfirmation: function(provider) {
		if ($.isFunction(global.ga)) {
			global.ga('send', 'event', 'Login', (provider || 'CT Network'),
					location.href);
		} else if (global._gaq) {
			global._gaq.push(['_trackEvent', 'Login', (provider || 'CT Network'),
					location.href]);
		}

		this.showConfirmation(messages.success, true);
	},

	/**
	 * Displays a "password retrieval email sent" message.
	 *
	 * @param email The email address to which the email was sent.
	 */
	showPasswordConfirmation: function(email) {
		var message = messages.retrievePassword.replace('{{email}}', email),
			$deck = this.$passwordDiv.find('.deck');

		this.$passwordDiv.find('.js-logInNow').removeClass('is-invisible');
		this.$passwordDiv.find('form').addClass('is-invisible');
		$deck.data('text', $deck.text()).text(message);
	},

	/* Determines which method should be called based on the the
	* action and response.
	*
	* @param action The action associated with the form. Either 'login'
	*               or 'forgotpassword'.
	* @param obj The object of JSON data returned from the server.
	*/
	routeResponse: function(action, obj) {
		var $form,
			message;

		if (action === 'forgotpassword') {

			if (obj.success) {
				this.showPasswordConfirmation(obj.email);
			} else {
				this.showError(messages.nonExistentEmail, this.$passwordForm);
			}
		} else {

			if (obj.isLoggedIn) {
				this.showLoginConfirmation();
			} else if (obj.needsToRegister) {
				this.events.trigger('login.newUser', {
					email: obj.email
				});
			} else {
				this.showError(messages.fail, this.$loginForm);
			}
		}
	},

	/**
	 * Sends an AJAX request to either log the user in, or handle the
	 * 'forgot password' request, based on the passed-in action. A
	 * loader/spinner will be displayed when the request is sent
	 * and will be hidden when the request is completed.
	 *
	 * @param action The action associated with the form. Either 'login'
	 *               or 'forgotpassword'.
	 * @param data The base query string to be passed to the server.
	 */
	send: function(action, data, ctMoviesVisibility) {
		//data += '&action=' + action;

		$.ajax({
			method: 'post',
			url: '/system/social/ajax/login.html',
			data: data,
			beforeSend: function(xhr, settings) {
				this.showLoader(this.$container.css('opacity', 0.5));
			}.bind(this),
			success: function(data, status, xhr) {
				this.$container.css('opacity', 1);
				this.hideLoader();
				this.routeResponse(action, $.parseJSON(xhr.responseText));
			}.bind(this),
			error: function(xhr, type, error) {
				this.hideLoader();
				this.showAjaxError(action, error);
			}.bind(this)
		});
	}
});

})(window.Zepto || window.jQuery, window, document);
/**
 * If this ever gets more complicated, then at some point it might make
 * sense to refactor this into separate MVC components. But right now
 * that seems a bit overkill.
 */
(function($, global, document) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});
cti.userLogin.login || (cti.userLogin.login = {});

var ns = cti.userLogin.login,
	messages = {
		fullName: 'Please enter your full name.',
		email: 'Please enter a valid email address',
		password: 'Please enter a password. Passwords cannot contain' +
				' the following characters: &, <, or >.',
		emailRegistered: 'It looks like that email address is already taken.' +
				' Please either enter the password associated with that account,' +
				' or choose a different email.',
		mismatchedEmail: 'The email addresses you entered do not match.',
		mismatchedPassword: 'The passwords you entered do not match.',
		ajaxError: 'Sorry, there was a problem with your request.' +
				' Please try again.',
		almostThere: '<strong>Almost There:</strong> You\'ve successfully' +
				' authenticated with %%provider%%. To finish linking your' +
				' accounts, please log in or register now.'
	};

ns.register = $.extend(Object.create(u$.loaderMixin), {

	// This object exists solely to provide `u$.loaderMixin` with the
	// CSS class it should apply to the loader HTML.
	options: {
		loaderClass: 'loader-login'
	},

	initialize: function($container, events) {
		if (!$container) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing widget container in `register` controller.'
			};
		}

		if (!events) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing events object in `register` controller.'
			};
		}

		this.events = events;

		this.setElements($container);
		this.subscribe();
		this.attach();
	},

	/**
	 * Sets the `$` HTML elements used by the application.
	 *
	 * @param $container The `$` HTML element container for the entire section.
	 */
	setElements: function($container) {
		this.$container = $container;
		this.$form = this.$container.find('form');

		this.setFields();
	},

	/**
	 * The register controller listens for two events:
	 * 1. When a new user tries logging in ('login.newUser').
	 * 2. When a new user tries loggin in via a social media account.
	 *    ('janrain.newUser');
	 * When this happens, the registration form will be populated with
	 * as much data as possible, and the registration widget will be
	 * expanded.
	 */
	subscribe: function() {
		this.events.on('login.newUser', this.setFormData, this);
		this.events.on('janrain.newUser', this.showPartialComplete, this);
	},

	/**
	 * Manages the following events:
	 * 1. Submit: Registration Form
	 * 2. Click: Social Media Links
	 */
	attach: function() {
		var onClick = $.proxy(function(e) {
				var $clicked = $(e.currentTarget);

				e.preventDefault();
				this.socialRegister($clicked);
			}, this),
			onSubmit = $.proxy(function(e) {
				var $form = $(e.currentTarget);

				e.preventDefault();
				this.register($form);
			}, this);

		this.$container.on('click', '.js-socialLogin', onClick).
				on('submit','.js-register', onSubmit);
	},

	/**
	 * Validates the first and last names, email addresses, and passwords.
	 *
	 * @param data An array of data to validate. Should be in the same
	 *             format produced by the jQuery/Zepto `serializeArray` method.
	 *
	 * @returns Boolean `true` if the data are valid. Otherwise `false`.
	 */
	validate: function(data) {
		data = data.reduceRight(function(memo, field) {
			memo[field.name] = field.value;

			return memo;
		}, {});

		return [this.validateName(data), this.validateEmail(data),
				this.validatePassword(data)].every(function(isValid) {
					return isValid;
				});
	},

	/**
	 * Validates a user's first and last names. If validation fails,
	 * then an error message will be displayed above the "first name" field.
	 *
	 * @param The data object containing the user info.
	 *
	 * @returns Boolean `true` if the name is valid. Otherwise `false`.
	 */
	validateName: function(data) {
		var first = data.firstname.toLowerCase(),
			last = data.lastname.toLowerCase(),
			isValid = true;

		if (first === '' || first.indexOf('first name') !== -1) {
			isValid = false;
		}

		if (last === '' || last.indexOf('last name') !== -1) {
			isValid = false;
		}

		if (!isValid) {
			this.showError(messages.fullName, this.fields.$firstName);
		}

		return isValid;
	},

	/**
	 * Validates the "email" and "confirm email" fields. If validation fails,
	 * then an error message will be displayed above the field in question.
	 *
	 * @param The data object containing the user info.
	 *
	 * @returns Boolean `true` if the emails are valid. Otherwise `false`.
	 */
	validateEmail: (function() {
		var pattern = /^[A-Za-z0-9\-_\.]+@[A-Za-z0-9\-_\.]+$/;

		return function(data) {
			var isValid = true;

			if (!pattern.test(data.email)) {
				isValid = false;

				this.showError(messages.email, this.fields.$email);
			}

			if (!this.fields.$confirmEmail.hasClass('is-invisible') &&
					data.email.toLowerCase() !== data.confirmemail.toLowerCase()) {
				isValid = false;

				this.showError(messages.mismatchedEmail, this.fields.$email);
			}

			return isValid;
		};
	})(),

	/**
	 * Validates the "password" and "confirm password" fields. If validation
	 * fails, then an error message will be displayed above the field in
	 * question.
	 *
	 * @param The data object containing the user info.
	 *
	 * @returns Boolean `true` if the passwords are valid. Otherwise `false`.
	 */
	validatePassword: (function() {
		var pattern = /[^&<>]/g;

		return function(data) {
			var isValid = true;

			if (!pattern.test(data.password)) {
				isValid = false;

				this.showError(messages.password, this.fields.$password);
			}

			if (data.password !== data.confirmpassword) {
				isValid = false;

				this.showError(messages.mismatchedPassword, this.fields.$password);
			}

			return isValid;
		};
	})(),

	register: function($form) {
		var data = $form.serializeArray();

		if (this.validate(data)) {
			this.hideFormErrors();
			this.send($form.serialize());
		}
	},

	/**
	 * Hands control over the Janrain controller (not directly, but via
	 * the events object), passing it the provider name.
	 *
	 * @param $clicked The `$` HTML element that was clicked.
	 *
	 * @fires register.socialRegister (passed `provider`).
	 */
	socialRegister: function($clicked) {
		var provider = $clicked.data('provider');

		this.events.trigger('register.socialRegister', provider);
	},

	showPartialComplete: function(data) {
		var html = messages.almostThere.replace('%%provider%%', data.providerName),
			$expandedView = this.$container.find('.expandedView');

		if (!this.$almostThere) {
			this.$almostThere = $('<p class="almostThere" />')
					.appendTo($expandedView);
		}

		// 1) Hide the social media buttons and display "Almost There" text
		this.$almostThere.html(html);
		this.$container.find('.socialMediaLinks').addClass('is-invisible');
		$expandedView.css('background', '#fff');

		// 2 Set the current social media provider
		this._currentProvider = data.providerName;

		// 3) Update the inputs as needed, and hide the non-essential fields.
		this.updateFields(data, true);
	},

	/**
	 * Takes an object of user data and uses it to populate
	 * the registration form.
	 *
	 * @param data An object containing the user's first and last names,
	 *             email address, and social media provider and identifier.
	 */
	setFormData: (function() {
		var defaults = {
			firstname: null,
			lastname: null,
			email: null,
			identifier: null
		};

		return function(data) {
			data = $.extend({}, defaults, data);

			this.updateFields(data, true);
		};
	})(),

	/**
	 * Loads the DOM element fields from the registration form.
	 *
	 * @returns An object representing the field elements.
	 */
	setFields: function() {
		var $form = this.$form,
			fields = ('firstName,lastName,email,confirmEmail,password,' +
					'confirmPassword,zipCode,gender,identifier').split(',');

		this.fields = fields.reduceRight(function(memo, field) {
			memo['$' + field] = $form.find('.' + field);

			return memo;
		}, {});

		return this.fields;
	},

	/**
	 * Updates the input values of the registration form. The following
	 * fields can be updated: first name, last name, email, and social
	 * media account identifier (e.g., a link to the user's profile page).
	 * The CSS class `is-marked` will be added to any input not populated.
	 *
	 * @param data The object containing the form values.
	 * @param hideExtraFields Boolean indicating whether certain auxiliary
	 *                        fields should be hidden. See the comments on
	 *                        the `hideExtraFields` method.
	 */
	updateFields: function(data, hideExtraFields) {
		// hide any error messages that may have been displayed earlier.
		this.hideFormErrors();

		if (hideExtraFields) {
			this.hideExtraFields();
		}

		data.confirmemail = data.email;

		if (this.validateEmail(data)) {
			this.fields.$confirmEmail.addClass('is-invisible');
		}

		$.each(this.fields, function(name, $input) {
			var key;

			// Only alter elements not hidden above.
			if (!$input.hasClass('is-invisible')) {
				key = name.replace('$', '').toLowerCase();

				if (data[key]) {
					$input.val(data[key]);
				} else {
					$input.addClass('is-marked');
				}
			}
		});
	},

	/**
	 * Hides all the error messages for the registration form fields.
	 * There is one error message for both first name and last name,
	 * the 'fullName' error message.
	 */
	hideFormErrors: function() {
		// hide any authentication error associated with the form itself.
		this.hideError(this.$form);

		if (this.fields) {
			$.each(this.fields, function(key, value) {

				if (key !== 'lastName') {
					this.hideError(value);
				}
			}.bind(this));
		}
	},

	/**
	 * Displays an HTML error message to the user.
	 *
	 * @param message The string message to display.
	 * @param $input The `$` input element associated with the error.
	 */
	showError: function(message, $input) {
		var $error = $input.prev('.error');

		if (!$error.length) {
			$error = $('<p class="error" />');
			$input.before($error);
		}

		$error.css('opacity', 0).html(message).removeClass('is-invisible').
				animate({
					opacity: 1
				});
	},

	/**
	 * Hides an error message.
	 *
	 * @param $form The `$` input element associated with the error.
	 */
	hideError: function($input) {
		var $error = $input.prev('.error');

		if ($input.length) {
			$error.animate({
				opacity: 0
			}, {
				complete: function() {
					$error.addClass('is-invisible');
				}
			});
		}
	},

	/**
	 * Displays the default "AJAX Error" message, but also appends the
	 * error message provided by the XHR object assuming it's provided.
	 *
	 * @param error The error message returned by the XHR object.
	 */
	showAjaxError: function(error) {
		var message = messages.ajaxError;

		if (error) {
			message += '<br />Error: ' + error;
		}

		this.showError(message, this.$form);
	},

	/**
	 * Displays a login confirmation message to the user and sends a
	 * login event to Google Analytics.
	 *
	 * @param obj The JSON object returned from the server. Expects
	 *            `providerName` and `elements` (for HTML generation) parameters.
	 */
	showConfirmation: function(obj) {
		//var frag = ns.renderConfirmationHTML(obj.elements);

		if ($.isFunction(global.ga)) {
			global.ga('send', 'event', 'Register', (this._currentProvider ||
					'CT Network'), location.href);
		} else if (global._gaq) {
			global._gaq.push(['_trackEvent', 'Register', (this._currentProvider ||
					'CT Network'), location.href]);
		}

		// display the message;
		this.events.trigger('register.showConfirmation', obj.elements);
	},

	/**
	 * In order to discourage users from abandoning the registration form,
	 * certain non-required fields can be hidden. Currently, these include
	 * the "Zip Code" and "Gender" fields.
	 */
	hideExtraFields: function() {
		this.fields.$zipCode.addClass('is-invisible');
		this.fields.$gender.addClass('is-invisible');
	},

	/* Determines which method should be called based on the the
	* AJAX response (object).
	*
	* @param obj The object of JSON data returned from the server.
	*/
	routeResponse: function(obj) {
		var $form,
			message;

		if (obj.isLoggedIn) {
			this.showConfirmation(obj);
		} else {
			message = obj.authenticate_error || messages.emailRegistered;

			this.showError(message, this.$form);
		}
	},

	send: function(data) {
		$.ajax({
			method: 'post',
			url: '/system/social/ajax/login.html',
			data: data,
			beforeSend: function(xhr, settings) {
				this.showLoader(this.$container.css('opacity', 0.5));
			}.bind(this),
			success: function(data, status, xhr) {
				this.$container.css('opacity', 1);
				this.hideLoader();
				this.routeResponse($.parseJSON(xhr.responseText));
			}.bind(this),
			error: function(xhr, type, error) {
				this.hideLoader();
				this.showAjaxError(error);
			}.bind(this)
		});
	}
});

})(window.Zepto || window.jQuery, window, document);
/**
 * If this ever gets more complicated, then at some point it might make
 * sense to refactor this into separate MVC components. But right now
 * that seems a bit overkill.
 */
(function($, global, document) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});
cti.userLogin.login || (cti.userLogin.login = {});

var ns = cti.userLogin.login,
	messages = {
		invalidAccount: 'The account number you entered was in the correct' +
				' format. Your account number will begin with three letters and' +
				' be followed by a string of numbers.',
		ajaxError: 'Sorry, there was a problem with your request.' +
				' Please try again.'
	};

ns.activate = $.extend(Object.create(u$.loaderMixin), {

	// This object exists solely to provide `u$.loaderMixin` with the
	// CSS class it should apply to the loader HTML.
	options: {
		loaderClass: 'loader-login'
	},

	initialize: function($container, events) {
		if (!$container) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing widget container in `activate` controller.'
			};
		}

		if (!events) {
			throw {
				name: 'JanrainLoginError',
				message: 'Missing events object in `activate` controller.'
			};
		}

		this.events = events;

		this.setElements($container);
		this.attach();
	},

	/**
	 * Sets the `$` HTML elements used by the application.
	 *
	 * @param $container The `$` HTML element container for the entire section.
	 */
	setElements: function($container) {
		this.$container = $container;
		this.$form = this.$container.find('form');
	},

	/**
	 * Manages the submit event for "activate subscription" form.
	 */
	attach: function() {
		var onSubmit = $.proxy(function(e) {
				var $form = $(e.currentTarget);

				e.preventDefault();

				// Send the password recovery email, or attempt to log the user in,
				// depending on the form element submitted.
				this.activate($form);
			}, this);

		this.$container.find('form').on('submit', onSubmit);
	},

	validate: (function() {
		var pattern = /[a-z]{3}\d+/i;

		return function(data) {
			data = data.filter(function(item) {
				return item.name === 'cdsaccountnumber';
			});

			return pattern.test(data[0].value);
		};
	})(),

	activate: function($form) {
		if (!this.validate($form.serializeArray())) {
			this.showError(messages.invalidAccount, $form);
		} else {
			this.hideError($form);
			this.send($form.serialize());
		}
	},

	/**
	 * Displays an HTML error message to the user.
	 *
	 * @param message The string message to display.
	 * @param $input The `$` input element associated with the error.
	 */
	showError: function(message, $input) {
		var $error = $input.prev('.error');

		if (!$error.length) {
			$error = $('<p class="error" />');
			$input.before($error);
		}

		$error.css('opacity', 0).html(message).removeClass('is-invisible').
				animate({
					opacity: 1
				});
	},

	/**
	 * Displays the default "AJAX Error" message, but also appends the
	 * error message provided by the XHR object assuming it's provided.
	 *
	 * @param error The error message returned by the XHR object.
	 */
	showAjaxError: function(error) {
		var message = messages.ajaxError;

		if (error) {
			message += '<br />Error: ' + error;
		}

		this.showError(message, this.$form);
	},

	/**
	 * Hides an error message.
	 *
	 * @param $form The `$` form element associated with the error.
	 */
	hideError: function($form) {
		var $error = $form.prev('.error');

		if ($error.length) {
			$error.animate({
				opacity: 0
			}, {
				complete: function() {
					$error.addClass('is-invisible');
				}
			});
		}
	},

	/**
	 * Displays an activation confirmation message to the user and sends a
	 * login event to Google Analytics.
	 *
	 * @param obj The JSON object returned from the server. Expects an
	 *            `elements` parameter for HTML generation.
	 */
	showConfirmation: function(obj) {
		if ($.isFunction(global.ga)) {
			global.ga('send', 'event', 'Activate', location.href);
		} else if (global._gaq) {
			global._gaq.push(['_trackEvent', 'Activate', location.href]);
		}

		// display the message;
		this.events.trigger('activate.showConfirmation', obj.elements);
	},

	/* Determines which method should be called based on the the
	* AJAX response (object).
	*
	* @param obj The object of JSON data returned from the server.
	*/
	routeResponse: function(obj) {
		if (!obj.success) {
			this.showError(obj.error, this.$form);
		} else {
			this.showConfirmation(obj);
		}
	},

	send: function(data) {
		$.ajax({
			method: 'post',
			url: '/system/social/ajax/login.html',
			data: data,
			beforeSend: function(xhr, settings) {
				this.showLoader(this.$container.css('opacity', 0.5));
			}.bind(this),
			success: function(data, status, xhr) {
				this.$container.css('opacity', 1);
				this.hideLoader();
				this.routeResponse($.parseJSON(xhr.responseText));
			}.bind(this),
			error: function(xhr, type, error) {
				this.hideLoader();
				this.showAjaxError(error);
			}.bind(this)
		});
	}
});

})(window.Zepto || window.jQuery, window, document);
/**
 * This is the root controller for the Janrain login.
 * It adds event listeners to load and display the modal, and
 * also is responsible for instantiating the other controllers
 * (e.g., janrain, login, register, and activate).
 */
(function($, global, document) {
'use strict';

global.cti || (global.cti = {});
cti.userLogin || (cti.userLogin = {});
cti.userLogin.login || (cti.userLogin.login = {});

var ns = cti.userLogin.login,
	defaults = {
		// startMessage,

		startOpen: 'login'
	};

function factory(proto) {
	var instance = Object.create(proto);

	if ($.isFunction(instance.initialize)) {
		instance.initialize.apply(instance, [].slice.call(arguments, 1));
	}

	return instance;
}

ns.controller = {

	/**
	 * Creates the modal instance, which will be loaded and displayed
	 * when users click links with the class `.js-login`.
	 */
	initialize: function() {
		this.options = $.extend({}, defaults, (ns.options || null));
		this.events = Object.create(cti.userLogin.events);

		this.subscribe();
		this.setModal();
	},

	subscribe: function() {
		this.events.on('register.showConfirmation', this.showConfirmation, this);
		this.events.on('activate.showConfirmation', this.showConfirmation, this);
	},

	setModal: function() {
		var $body = $(document.body),
			args = [{
				url: '/system/social/ajax/get_login.html',
				query: function($trigger) {
					var ctMoviesCheck = $("#ctmovies"), // ID to check faster
						ctMoviesVisibility = false,
						itemName = $trigger.data('itemname');
						
					this.setStartState($trigger);
					
					if(ctMoviesCheck.length===0){
						return "page=" + encodeURIComponent(location.href) + "&start=" + this.options.startOpen + "&itemName=" + itemName
					}else{
						return "page=" + encodeURIComponent(location.href) + "&ctmovies=true&start=" + this.options.startOpen
					}

				}.bind(this),
				triggerClass: 'js-login',
				destroyOnClose: false,
				cache: true,
				clickOverlayToClose: false,
				beforeShow: function($modal, $trigger) {
					$modal.attr('id', 'ctSocialLogin');

					// keep track of the clicked element's data
					this.setDataFromTrigger($trigger);
				}.bind(this),
				show: function($modal) {
					this.$modal = $modal.addClass('is-centered');
					this.loadWidgets($modal);
					this.subscribe();
					this.setCTNetworkTips();
					this.$modal.find(".js-email").focus();
					// If the `displayOnLoad` option is set, then the modal will be
					// displayed immediately. However, once it is displayed, users
					// will need a way to reopen it once it has been closed.
					if (this.options.displayOnLoad) {
						$modal.attach($body);
					}
				}.bind(this)
			}];

		if (!this.options.displayOnLoad) {
			args.unshift($body);
		}

		this.modal = u$.modal.apply(null, args);
	},

	setDataFromTrigger: function($trigger) {
		var data = {};

		if ($trigger.length) {
			data.redirectUrl = {
				hash: $trigger.data('loginhash') || ''
			};
		}

		ns._startData = data;
	},

	/**
	 * This is a temporary until we have a better
	 * tooltip function/plugin in place.
	 */
	setCTNetworkTips: (function() {
		var timer;

		function setTimer($tip) {
			timer = setTimeout(function() {
				$tip.animate({
					opacity: 0
				}, {
					complete: function() {
						$tip.addClass('is-invisible');
					}
				});
			}, 300);
		}

		return function() {
			var $tip = $('#ctSocialLoginNetworkInfo');

			$tip.on({
				mouseenter: function() {
					clearTimeout(timer);
				},
				mouseleave: function(e) {
					setTimer($tip);
				}
			});

			this.$modal.on({
				'mouseenter': function(e) {
					var $trigger = $(e.target);

					$tip.appendTo(document.body).css({
						position: 'absolute',
						left: $trigger.offset().left,
						top: $trigger.offset().top + 20,
						opacity: 0
					}).removeClass('is-invisible').animate({
						opacity: 1
					});
				},

				'mouseleave': function() {
					setTimer($tip);
				}
			}, '.js-tooltip');
		};
	})(),

	/**
	 * Determines which widget should be open and whether a message
	 * message should be displayed when the modal loads.
	 *
	 * @param $trigger The `$` element that was clicked. This element
	 *                 can have `data-message` and `data-start` attributes.
	 */
	setStartState: function($trigger) {
		var message,
			href,
			start;

		if ($trigger && $trigger.length) {
			message = $trigger.data('message');
			start = $trigger.data('start');

			if (message) {
				this.options.startMessage = message;
			}

			if (start === 'register' || 'activate') {
				this.options.startOpen = start;
			}
		}
	},

	/**
	 * Instantiates the janrain, login, register, activate, and accordion
	 * controllers, in that order.
	 *
	 * @param $modal The login $modal window
	 */
	loadWidgets: function($modal) {
		var ctors = 'login,register,activate'.split(','),
			args;

		args = ctors.map(function(proto) {
			var $container = $modal.find('.js-' + proto + 'Widget');

			if ($container.length) {
				this[proto] = factory(ns[proto], $container, this.events);
			}
			
			return $container.length ? $container : null;
		}, this);

		// factory is basically called as follows:
		// `factory(ns.accordion, login, register, activate);`
		// where the first arg is a prototype and the others
		// arguments are object instances.
		this.janrain = factory(ns.janrain, this.events);
		args.unshift(ns.accordion, this.events, $modal);
		this.accordion = factory.apply(null, args);
		this.accordion.expand(this.options.startOpen);
	},

	showConfirmation: function(data) {
		var $modal = this.$modal,
			$conf = $modal.find('#ctSocialLoginConfirmation'),
			$children = $modal.find('.js-hideOnLogin'),
			frag = ns.renderConfirmationHTML(data),
			l = $children.length - 1;

		$conf.html(frag).addClass('widget').css('opacity', 0).
				removeClass('is-invisible');
		$children.slideUp({
			duration: 300,
			complete: function() {
				$(this).addClass('is-invisible');
			}
		}).animate({
			opacity: 0
		});
		$conf.animate({
			opacity: 1
		});

		this.modal.detach();
	}
}; 

/**
 * Sets the front controller. Fired when the Janrain
 * library is loaded.
 */
window.janrainWidgetOnload = function() {
	ns.package = factory(ns.controller);
};

})(window.Zepto || window.jQuery, window, document);
(function() {
	'use strict';

	// load the CSS file
	var css = document.createElement('link'),
		head = document.getElementsByTagName('head')[0];
	css.rel = 'stylesheet';
	css.href = '/system/social/login/src/css/login.css?08282014';

	head && head.parentNode.appendChild(css);

	if (typeof window.janrain !== 'object') window.janrain = {};
	if (typeof window.janrain.settings !== 'object') window.janrain.settings = {};
	
	// Enable custom configuration
	janrain.settings.custom = true;
	// The token url won't be used, but is still required by janrain for legacy reasons
	// Use location.hostname so that the report in rpxnow.com is displayed correctly
	janrain.settings.tokenUrl = 'http://' + location.hostname + '/system/social/ajax.html?janrain=true';
	// tell janrain to look for the janrainWidgetOnload function
	janrain.settings.tokenAction = 'event';
	
	function isReady() {
		janrain.ready = true;
	}
	
	if (document.addEventListener) {
		document.addEventListener("DOMContentLoaded", isReady, false);
	} else {
		window.attachEvent('onload', isReady);
	}
	
	var e = document.createElement('script');
	e.type = 'text/javascript';
	e.id = 'janrainAuthWidget';
	
	if (document.location.protocol === 'https:') {
		e.src = 'https://rpxnow.com/js/lib/login.christianitytoday.com/engage.js';
	} else {
		e.src = 'http://widget-cdn.rpxnow.com/js/lib/login.christianitytoday.com/engage.js';
	}
	
	var s = document.getElementsByTagName('script')[0];
	//s.parentNode.insertBefore(e, s);
	document.body.appendChild(e);
})();